<?php
// public/login_action.php
declare(strict_types=1);

/* -------- Unified Session Setup -------- */
require __DIR__ . '/session_boot.php';
/* -------------------------------------- */

/* Graceful error handler → redirect with message instead of 500 */
function fail(string $msg): void {
  header('Location: executive_login.php?error=' . urlencode($msg));
  exit;
}

/* Safe DB include (no guards). Fallback to db.php if needed */
$pdo = null;
try {
  $dbPdoOnly = __DIR__ . '/../includes/db_pdo_only.php';
  if (is_file($dbPdoOnly)) {
    require $dbPdoOnly;
  } else {
    require __DIR__ . '/../includes/db.php';
  }
} catch (Throwable $e) {
  fail('Database error: ' . $e->getMessage());
}

// DO NOT include helpers here (they may redirect during login)
// require __DIR__ . '/../includes/helpers.php';

/** Resolve assignment from up_admin_tree.php */
function resolve_assignment_from_tree(string $stateKey, string $nodalKey, string $districtKey, string $ulbName): ?array {
  $treePath = __DIR__ . '/up_admin_tree.php';
  if (!is_file($treePath)) return null;
  $TREE = include $treePath;

  if (empty($TREE[$stateKey])) return null;
  $state      = $TREE[$stateKey];
  $state_name = $state['name'] ?? $stateKey;
  $state_code = $state['code'] ?? '';

  $nodals = $state['nodals'] ?? [];
  if (empty($nodals[$nodalKey])) return null;
  $nodal      = $nodals[$nodalKey];
  $nodal_code = $nodal['code'] ?? '';

  $districts = $nodal['districts'] ?? [];
  if (empty($districts[$districtKey])) return null;
  $district      = $districts[$districtKey];
  $district_code = $district['code'] ?? '';

  $ulb_code = '';
  foreach (($district['ulbs'] ?? []) as $u) {
    if (isset($u['name']) && $u['name'] === $ulbName) { $ulb_code = $u['code'] ?? ''; break; }
  }
  if ($ulb_code === '') return null;

  return [
    'state_name'    => $state_name,
    'state_code'    => $state_code,
    'nodal_name'    => $nodalKey,
    'nodal_code'    => $nodal_code,
    'district_name' => $districtKey,
    'district_code' => $district_code,
    'ulb_name'      => $ulbName,
    'ulb_code'      => $ulb_code,
  ];
}

/* Input */
$phone    = trim($_POST['phone'] ?? '');
$inputPwd = (string)($_POST['password'] ?? '');
if (!preg_match('/^\d{10}$/', $phone) || $inputPwd === '') {
  fail('Enter 10-digit phone and password.');
}

try {
  // Fetch user
  $sql = 'SELECT id, phone, email, full_name, address,
                 home_district, duty_state, duty_nodal, duty_district, duty_ulb,
                 password
          FROM executives
          WHERE phone = :p
          LIMIT 1';
  $st = $pdo->prepare($sql);
  $st->execute(['p' => $phone]);
  $user = $st->fetch(PDO::FETCH_ASSOC);

  if (!$user) {
    fail('Invalid phone or password.');
  }

  $storedPlain = (string)($user['password'] ?? '');
  if (!hash_equals($storedPlain, $inputPwd)) {
    fail('Invalid phone or password.');
  }

  // Success → set session
  session_regenerate_id(true);
  $_SESSION['user_id'] = (int)$user['id'];

  // Resolve assignment
  $resolved = resolve_assignment_from_tree(
    (string)$user['duty_state'],
    (string)$user['duty_nodal'],
    (string)$user['duty_district'],
    (string)$user['duty_ulb']
  );
  if (!$resolved) {
    fail('Duty assignment not found in admin tree.');
  }

  $_SESSION['assignment'] = $resolved + [
    'state'    => $resolved['state_name'],
    'nodal'    => $resolved['nodal_name'],
    'district' => $resolved['district_name'],
    'ulb'      => $resolved['ulb_name'],
  ];

  // Go to Attendance first
  header('Location: attendance.php');
  exit;

} catch (Throwable $e) {
  fail('Login error: ' . $e->getMessage());
}
