// public/assets/app.js
document.addEventListener('DOMContentLoaded', () => {
    
    
  const $  = (s,ctx=document)=>ctx.querySelector(s);
  const $$ = (s,ctx=document)=>Array.from(ctx.querySelectorAll(s));
  function normalizeUrl(u){
  if (!u) return '';
  // force https on https pages
  if (location.protocol === 'https:' && u.startsWith('http://')) {
    u = 'https://' + u.slice(7);
  }
  // ensure /sbn prefix when path starts with /waste_seg_form...
  if (u.startsWith('/waste_seg_form/')) {
    u = '/sbn' + u;
  }
  return u;
}

  // ---- MOBILE DEBUG HUD ----
(function(){
  const bar = document.createElement('div');
  bar.id = 'mdbg';
  bar.style.cssText =
    'position:fixed;left:0;right:0;bottom:0;z-index:99999;background:#111;color:#0f0;' +
    'font:12px/1.4 monospace;padding:6px 8px;max-height:40vh;overflow:auto;opacity:.95';
  document.addEventListener('DOMContentLoaded', ()=>document.body.appendChild(bar));
  function log(msg){ try{
    const p=document.createElement('div'); p.textContent=String(msg);
    bar.appendChild(p);
  }catch(_){}
  }
  window.mdbg = log;
  window.addEventListener('error', e=>log('JS error: ' + (e.message||e)));
  window.addEventListener('unhandledrejection', e=>{
    const r = e.reason;
    log('Promise error: ' + (r && (r.message||r.stack||r)) );
  });
})();


  function getGeo() {
  return new Promise((resolve) => {
    if (!navigator.geolocation) return resolve(null);
    navigator.geolocation.getCurrentPosition(
      pos => resolve({
        lat: pos.coords.latitude,
        lng: pos.coords.longitude,
        acc: pos.coords.accuracy
      }),
      () => resolve(null),
      { enableHighAccuracy: true, timeout: 9000, maximumAge: 0 }
    );
  });
}
function flagEmoji(cc) {
  if (!cc || cc.length!==2) return '';
  return String.fromCodePoint(...cc.toUpperCase().split('').map(c => 127397 + c.charCodeAt()));
}
async function reverseGeocode(lat, lng) {
  const token = (window.APP_CFG && window.APP_CFG.MAPBOX_TOKEN) || '';
  const gkey  = (window.APP_CFG && window.APP_CFG.GOOGLE_KEY) || '';
  // Try Mapbox first
  if (token) {
    try {
      const url = `https://api.mapbox.com/geocoding/v5/mapbox.places/${lng},${lat}.json?language=en&types=place,locality,neighborhood,district,region,country,address,postcode&access_token=${encodeURIComponent(token)}`;
      const r = await fetch(url);
      const j = await r.json();
      const feat = (j && j.features) ? j.features : [];
      const ctx = (feat[0] && feat[0].context) ? feat[0].context : [];
      const getBy = (idPrefix) => {
        const fromMain = feat.find(f => (f.id||'').startsWith(idPrefix));
        if (fromMain) return { text: fromMain.text, code: fromMain.short_code || '' };
        const c = ctx.find(x => (x.id||'').startsWith(idPrefix));
        return c ? { text: c.text, code: c.short_code || '' } : null;
      };
      const village  = (feat.find(f => ['place','locality','neighborhood'].some(p=> (f.id||'').startsWith(p))) || {}).text || '';
      const district = (getBy('district') || {}).text || '';
      const stateObj = getBy('region');
      const state    = stateObj ? stateObj.text : '';
      const countryObj = getBy('country');
      const country  = countryObj ? countryObj.text : '';
      let cc = countryObj ? (countryObj.code || '') : '';
      cc = cc ? cc.split('-').pop().toUpperCase() : '';
      const full = (feat[0] && feat[0].place_name) ? feat[0].place_name : '';
      return { village, district, state, country, cc, address: full };
    } catch(e) { /* fallthrough to Google */ }
  }
  // Fallback Google (needs paid key/billing)
  if (gkey) {
    try {
      const url = `https://maps.googleapis.com/maps/api/geocode/json?latlng=${lat},${lng}&key=${encodeURIComponent(gkey)}`;
      const r = await fetch(url);
      const j = await r.json();
      if (j.status === 'OK' && j.results && j.results.length) {
        const comps = j.results[0].address_components || [];
        const pick = (type) => {
          const c = comps.find(x => (x.types||[]).includes(type));
          return c ? c.long_name : '';
        };
        const village  = pick('sublocality') || pick('locality') || pick('administrative_area_level_3');
        const district = pick('administrative_area_level_2');
        const state    = pick('administrative_area_level_1');
        const country  = pick('country');
        const cc       = (comps.find(x => (x.types||[]).includes('country')) || {}).short_name || '';
        const address  = j.results[0].formatted_address || '';
        return { village, district, state, country, cc, address };
      }
    } catch(e) {}
  }
  return { village:'', district:'', state:'', country:'', cc:'', address:'' };
}
function drawStampBlock(ctx, canvas, lines) {
  const pad = 12, lineH = 20, font = '14px Poppins, Arial, sans-serif';
  ctx.font = font; ctx.textBaseline = 'top';
  const widths = lines.map(l => ctx.measureText(l).width);
  const w = Math.min(Math.max(...widths) + pad*2, canvas.width - 30);
  const h = lineH * lines.length + pad*2;
  const x = pad, y = canvas.height - h - pad;
  ctx.fillStyle = 'rgba(0,0,0,0.55)'; ctx.fillRect(x, y, w, h);
  ctx.fillStyle = '#fff';
  lines.forEach((line, i) => ctx.fillText(line, x + pad, y + pad + i*lineH));
}
function gmtOffsetString(date) {
  const m = -date.getTimezoneOffset(); // minutes ahead of UTC
  const sign = m >= 0 ? '+' : '-';
  const abs = Math.abs(m); const hh = String(Math.floor(abs/60)).padStart(2,'0'); const mm = String(abs%60).padStart(2,'0');
  return `GMT${sign}${hh}:${mm}`;
}


  /* ---------- Top message helpers ---------- */
  function showTopMessage(text, isError = true) {
    const top = $('#top-error');
    if (!top) return;
    top.style.display = '';
    top.classList.remove('visible','success','error');
    top.classList.add('visible', isError ? 'error' : 'success');
    top.textContent = text;
    if (isError) setTimeout(()=>{ top.style.display='none'; top.classList.remove('visible','error'); }, 6000);
  }

  async function postFormAjax(form) {
    const action = form.getAttribute('action') || 'submit.php';
    const fd = new FormData(form);
    const res = await fetch(action, { method: 'POST', body: fd, credentials: 'same-origin' });
    const ct = (res.headers.get('content-type')||'').toLowerCase();
    if (ct.includes('application/json')) return res.json();
    const text = await res.text();
    try { return JSON.parse(text); } catch { return { success:false, message:text }; }
  }

  /* ---------- Phone availability hint (registration) ---------- */
  const phone = $('#phone'), phoneMsg = $('#phone-msg');
  const getDetailsBtn = $('#get-details'); // <--- NEW: element for already-registered quick link

  // helper: run check for a phone string (returns parsed json or null)
  async function performPhoneCheck(v) {
    try {
      const res = await fetch('check_phone.php?phone='+encodeURIComponent(v),{cache:'no-store'});
      const j = await res.json();
      return j;
    } catch (e) {
      window.mdbg && mdbg('phone check fetch failed: ' + (e && (e.message||e)));
      return null;
    }
  }

  // core UI update using response object
  function updatePhoneUI(j) {
    if (!phoneMsg || !getDetailsBtn) return;
    if (!j) { phoneMsg.textContent = ''; getDetailsBtn.style.display='none'; return; }
    phoneMsg.textContent = j.exists ? 'Phone already registered' : 'Phone available';
    if (j.exists && j.id) {
      getDetailsBtn.style.display = 'inline-block';
      getDetailsBtn.href = 'officer.php?id=' + encodeURIComponent(j.id);
    } else {
      getDetailsBtn.style.display = 'none';
    }
  }

  if (phone) {
    let t;
    const scheduleCheck = (v) => {
      clearTimeout(t);
      t = setTimeout(async ()=>{
        if (!v || v.length!==10) { updatePhoneUI(null); return; }
        const j = await performPhoneCheck(v);
        updatePhoneUI(j);
      },400);
    };

    phone.addEventListener('input', ()=> {
      const v = phone.value.replace(/\D/g,'');
      scheduleCheck(v);
    });

    // Also run check on blur (for paste+tab) and immediately on load if field has 10 digits
    phone.addEventListener('blur', async ()=> {
      const v = phone.value.replace(/\D/g,'');
      if (v.length===10) {
        const j = await performPhoneCheck(v);
        updatePhoneUI(j);
      }
    });

    // initial immediate check if phone already has value on page load
    (async ()=> {
      const v = phone.value.replace(/\D/g,'');
      if (v.length===10) {
        const j = await performPhoneCheck(v);
        updatePhoneUI(j);
      }
    })();

    // ensure getDetailsBtn actually navigates (in case href="#" was used)
    if (getDetailsBtn) {
      getDetailsBtn.addEventListener('click', function(e){
        const href = this.getAttribute('href') || '';
        if (!href || href === '#') {
          e.preventDefault();
          // try to read id from href stored earlier or from check
          const v = phone.value.replace(/\D/g,'');
          if (v.length===10) {
            // try direct lookup to get id & navigate
            (async ()=>{
              const j = await performPhoneCheck(v);
              if (j && j.exists && j.id) {
                window.location.href = 'officer.php?id=' + encodeURIComponent(j.id);
              } else {
                showTopMessage('No resident found for this phone', true);
              }
            })();
          }
        } else {
          // normal link behaviour
        }
      });
    }
  }

  /* ---------- Camera helper (reusable) ---------- */
  function setupCamera(openSel, takeSel, retakeSel, videoSel, canvasSel, previewSel, photoInputSel) {
    const openBtn = $(openSel), takeBtn = $(takeSel), retakeBtn = $(retakeSel);
    const video = $(videoSel), canvas = $(canvasSel), preview = $(previewSel), photoInput = $(photoInputSel);
    let stream = null;

    async function start() {
      if (!navigator.mediaDevices || !navigator.mediaDevices.getUserMedia) { alert('Camera not supported'); return; }
      try {
        stream = await navigator.mediaDevices.getUserMedia({video:{facingMode:'environment'}});
        if (video) { video.srcObject = stream; video.style.display='block'; video.play().catch(()=>{}); }
        if (takeBtn) takeBtn.style.display='';
        if (openBtn) openBtn.style.display='none';
      } catch (e) { alert('Camera error: '+(e.message||e)); }
    }
    function stop() {
      if (!stream) return;
      stream.getTracks().forEach(t=>t.stop());
      stream=null;
      if (video) { video.pause(); video.srcObject=null; video.style.display='none'; }
      if (openBtn) openBtn.style.display='';
    }
    async function snap() {
    if (!video || !canvas) return;

    // draw current frame
     canvas.width = video.videoWidth || 1280;
     canvas.height = video.videoHeight || 720;
     const ctx = canvas.getContext('2d');
     ctx.drawImage(video, 0, 0, canvas.width, canvas.height);

  // capture geo + reverse geocode
  const geo = await getGeo();
  let lat=null, lng=null, acc=null, where={village:'',district:'',state:'',country:'',cc:'',address':''};

  if (geo) {
    lat = geo.lat; lng = geo.lng; acc = geo.acc;
    where = await reverseGeocode(lat, lng);
  }

  // format time
  const now = new Date();
  const dateStr = now.toLocaleDateString('en-IN');
  const timeStr = now.toLocaleTimeString('en-IN', { hour12:true });
  const gmtStr  = gmtOffsetString(now);
  const flag    = where.cc ? (' ' + flagEmoji(where.cc)) : '';

  // build 4 rows exactly as requested
  const line1 = `${where.village || '—'}, ${where.district || '—'}, ${where.state || '—'}, ${where.country || '—'}${flag}`;
  const line2 = where.address || 'Address unavailable';
  const line3 = (lat && lng) ? `Lat ${lat.toFixed(6)}  Long ${lng.toFixed(6)}` : 'Lat/Long unavailable';
  const line4 = `${dateStr}, ${timeStr}, ${gmtStr}`;

  drawStampBlock(ctx, canvas, [line1, line2, line3, line4]);

  // export and preview
  const data = canvas.toDataURL('image/jpeg', 0.9);
  if (photoInput) photoInput.value = data;
  if (preview)   preview.innerHTML = `<img src="${data}" style="max-width:320px;border-radius:8px">`;

  // push hidden fields (both pages supported)
  const set = (id, val) => { const el = document.getElementById(id); if (el) el.value = val; };
  set('captured_at', now.toISOString());  set('captured_at_off', now.toISOString());
  if (lat!=null && lng!=null) {
    set('geo_lat', lat); set('geo_lng', lng); set('geo_acc', acc||'');
    set('geo_lat_off', lat); set('geo_lng_off', lng); set('geo_acc_off', acc||'');
  }
  set('geo_addr', where.address);  set('geo_addr_off', where.address);
  set('geo_village', where.village); set('geo_district', where.district);
  set('geo_state', where.state);     set('geo_country', where.country); set('geo_cc', where.cc);
  set('geo_village_off', where.village); set('geo_district_off', where.district);
  set('geo_state_off', where.state);     set('geo_country_off', where.country); set('geo_cc_off', where.cc);

  // stop camera & toggle buttons
  stop();
  if (takeBtn)   takeBtn.style.display='none';
  if (retakeBtn) retakeBtn.style.display='';
}

    function retake() {
      if (preview) preview.innerHTML='';
      if (photoInput) photoInput.value='';
      if (retakeBtn) retakeBtn.style.display='none';
      start();
    }
    if (openBtn)  openBtn.addEventListener('click', start);
    if (takeBtn)  takeBtn.addEventListener('click', snap);
    if (retakeBtn)retakeBtn.addEventListener('click', retake);
  }

  // registration camera
  if ($('#open-camera')) {
    setupCamera('#open-camera','#take-photo','#retake-photo','#video','#canvas','#photo-preview','#photo_data');
  }
  // officer camera
  if ($('#open-camera-off')) {
    setupCamera('#open-camera-off','#take-photo-off','#retake-photo-off','#video','#canvas','#photo-preview-off','#photo_data_off');
  }

  /* ---------- Registration: prevent redirects with explicit button ---------- */
  {
    const regForm = $('#reg-form');
    const regBtn  = $('#save-register');
    if (regForm && regBtn) {
      regBtn.addEventListener('click', async () => {
        const ward  = $('#ward')?.value || '';
        const house = $('#house_no')?.value || '';
        const name  = $('#name')?.value || '';
        const phonev= ($('#phone')?.value||'').replace(/\D/g,'');
        const waste = $$('input[name="waste_segregation"]:checked').length;
        const photo = $('#photo_data')?.value || '';
        const date  = $('#date')?.value || '';
        const time  = $('#time')?.value || '';

        if (!date || !time || !ward || !house || !name || phonev.length!==10 || waste===0 || !photo) {
          showTopMessage('Please fill all required fields (including date/time) and take photo.', true);
          return;
        }

        regBtn.disabled = true; regBtn.textContent = 'Saving...';
        try {
          // use 'resp' as the response variable (consistent with officer handler)
          const resp = await postFormAjax(regForm);
          if (resp && resp.success) {
            // prefer server message, otherwise construct one from server fields or form
            let msg = resp.message;
            if (!msg) {
              const tn = resp.training_number || resp.next_training_number || $('#training_number')?.value;
              msg = tn ? `Training ${tn} data has been submitted` : 'Training data has been submitted';
            }
            showTopMessage(msg, false);
            setTimeout(()=> location.reload(), 1200);
          } else {
            showTopMessage(resp?.message || 'Save failed', true);
            regBtn.disabled = false; regBtn.textContent = 'Save';
          }
        } catch (e) {
          console.error(e);
          showTopMessage('Submission failed. Check console.', true);
          regBtn.disabled = false; regBtn.textContent = 'Save';
        }
      });
    }
  }

  /* ---------- Officer: lookup & render (hide initial if T1 exists) ---------- */
  // ---------- Officer: lookup & render ----------
  // ---------- Officer: lookup & render (robust) ----------
  (() => {
    // 1) ensure elements exist
    const btn     = document.querySelector('#lookup');
    const phoneEl = document.querySelector('#lookup_phone');
    const form    = document.querySelector('#lookup-form');
    if (!btn || !phoneEl) {
      window.mdbg && mdbg('Lookup wiring failed: #lookup or #lookup_phone not found');
      return;
    }

    // 2) ensure button doesn't submit form
    btn.setAttribute('type', 'button');

    // 3) helper: safe JSON parse
    const safeJson = async (res) => {
      const text = await res.text();
      try { return { ok: true, data: JSON.parse(text) }; }
      catch (e) { return { ok: false, text, err: e }; }
    };

    async function doLookup() {
      const p = (phoneEl.value || '').replace(/\D/g, '');
      if (p.length !== 10) { alert('Enter 10 digit phone'); return; }

      try {
        window.mdbg && mdbg('Lookup click -> phone=' + p);

        // if your public_lookup.php is inside /public, keep it RELATIVE
        const url = 'public_lookup.php?phone=' + p;

        const res = await fetch(url, { cache: 'no-store', credentials: 'same-origin' });
        window.mdbg && mdbg('HTTP ' + res.status + ' ' + res.statusText);

        const parsed = await safeJson(res);
        if (!parsed.ok) {
          window.mdbg && mdbg('JSON parse failed. First 200 chars:\n' + (parsed.text || '').slice(0,200));
          alert('Server did not return JSON (check PHP notices).');
          return;
        }

        const j = parsed.data;
        window.mdbg && mdbg('Parsed JSON: ' + JSON.stringify(j));

        if (!j || !j.found) { alert('Resident not found'); return; }

        const show = (sel, val) => { const el = document.querySelector(sel); if (el) el.textContent = val || ''; };
        const showImg = (sel, path) => {
          const el = document.querySelector(sel);
          if (!el) return;
          if (!path) { el.innerHTML=''; return; }
          // normalize minimal: force https if page is https and path starts with http://
          let u = path;
          if (location.protocol === 'https:' && /^http:\/\//i.test(u)) u = 'https://' + u.slice(7);
          if (u.startsWith('/waste_seg_form/')) u = '/sbn' + u; // your /sbn prefix rule
          el.innerHTML = `<img src="${u}" style="max-width:220px;border-radius:8px">`;
        };

        const box = document.querySelector('#resident-data');
        if (box) box.style.display = '';

        // populate ULB (NEW)
        show('#r_ulb', j.data.ulb);

        show('#r_ward', j.data.ward);
        show('#r_house_no', j.data.house_no);
        show('#r_name', j.data.name);
        show('#r_phone', j.data.phone);
        show('#r_waste', j.data.waste_segregation);

        showImg('#reg-photo', j.data.photo_path);

        const prev = document.querySelector('#previous-trainings');
        if (prev) prev.innerHTML = '';
        if (Array.isArray(j.trainings)) {
          j.trainings.forEach(t => {
            const d = document.createElement('div');
            const num = Number(t.training_number);
            const label = (num===1 ? 'Training First' : num===2 ? 'Training Second' : 'Training Third');
            let u = t.photo_path || '';
            if (u) {
              if (location.protocol === 'https:' && /^http:\/\//i.test(u)) u = 'https://' + u.slice(7);
              if (u.startsWith('/waste_seg_form/')) u = '/sbn' + u;
            }
            // include ULB in each training card (use t.ulb or fall back to resident ulb)
            d.innerHTML = `
              <strong>${label}</strong>
              <div>Date: ${t.training_date || ''}</div>
              <div>Waste: ${t.waste_segregation || ''}</div>
              <div>ULB: ${t.ulb || j.data.ulb || ''}</div>
              <div style="margin-top:6px">${u ? `<img src="${u}" style="max-width:220px;border-radius:6px">` : ''}</div>
              <hr>`;
            prev && prev.appendChild(d);
          });
        }

        // hide initial block if T1 exists
        const init = document.querySelector('#initial-block');
        if (init) {
          const hasT1 = (j.trainings || []).some(t => Number(t.training_number) === 1);
          init.style.display = hasT1 ? 'none' : '';
        }

        // next training
        const nlab = document.querySelector('#next-training-label');
        if (nlab) nlab.textContent = j.next_training_label || 'Training';

        const nextSec = document.querySelector('#next-training-section');
        if (j.next_training_number === null) {
          if (nextSec) nextSec.style.display = 'none';
        } else {
          if (nextSec) nextSec.style.display = '';
          const rid = document.querySelector('#resident_id');
          if (rid) rid.value = j.data.id;
          const pp = document.querySelector('#photo-preview-off');
          if (pp) pp.innerHTML = '';
          const pi = document.querySelector('#photo_data_off');
          if (pi) pi.value = '';
          document.querySelectorAll('#next-training-section input[name="waste_segregation"]').forEach(r=>r.checked=false);
          const d = new Date();
          const rdate = document.querySelector('#r_date');
          const rtime = document.querySelector('#r_time');
          if (rdate) rdate.value = d.toISOString().slice(0,10);
          if (rtime) rtime.value = d.toTimeString().slice(0,5);
        }
      } catch (e) {
        window.mdbg && mdbg('Lookup failed: ' + (e && (e.message || e)));
        alert('Lookup failed (see debug HUD).');
      }
    }

    // click + submit wiring
    btn.addEventListener('click', e => { e.preventDefault(); doLookup(); });
    form && form.addEventListener('submit', e => { e.preventDefault(); doLookup(); });

    // boot log (so we know script ran)
    window.mdbg && mdbg('Lookup wiring ready');
  })();


  /* ---------- Officer: explicit save button (no redirect) ---------- */
  {
    const offForm = $('#officer-form');
    const offBtn  = $('#save-training');
    if (offForm && offBtn) {
      offBtn.addEventListener('click', async ()=>{
        const rid   = $('#resident_id')?.value || '';
        const photo = $('#photo_data_off')?.value || '';
        const waste = $$('#next-training-section input[name="waste_segregation"]:checked').length;
        const d = $('#r_date')?.value || '', t = $('#r_time')?.value || '';

        if (!rid) { showTopMessage('Please lookup a resident first.', true); return; }
        if (!d || !t) { showTopMessage('Please set date/time.', true); return; }
        if (waste===0) { showTopMessage('Please choose waste segregation.', true); return; }
        if (!photo) { showTopMessage('Please capture photo for this visit.', true); return; }

        offBtn.disabled = true; offBtn.textContent = 'Saving...';
        try{
          const resp = await postFormAjax(offForm);
          if (resp && resp.success) {
            showTopMessage(resp.message || 'Saved', false);
            setTimeout(()=> location.reload(), 1200);
          } else {
            showTopMessage(resp?.message || 'Save failed', true);
            offBtn.disabled = false; offBtn.textContent = 'Save Training';
          }
        }catch(err){
          console.error(err);
          showTopMessage('Submission failed. Check console.', true);
          offBtn.disabled = false; offBtn.textContent = 'Save Training';
        }
      });
    }
  }

  /* ---------- init: hide empty error boxes ---------- */
  $$('.error').forEach(el => { el.style.display = 'none'; });
});
